<?php

namespace App\Http\Controllers;

use App\Models\Task;
use App\Models\TaskApproval;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class DashboardController extends Controller
{
    public function index()
    {
        $user = Auth::user();

        if ($user->isSuperAdmin() || $user->isAdmin()) {
            return $this->adminDashboard();
        }

        if ($user->isMarketing()) {
            return $this->marketingDashboard();
        }

        return $this->employeeDashboard();
    }

    private function adminDashboard()
    {
        $stats = [
            'total_tasks'    => Task::count(),
            'pending'        => Task::where('status', 'pending')->count(),
            'in_progress'    => Task::where('status', 'in_progress')->count(),
            'under_review'   => Task::where('status', 'under_review')->count(),
            'completed'      => Task::whereIn('status', ['completed', 'approved'])->count(),
            'total_users'    => User::where('role', '!=', 'super_admin')->count(),
        ];

        $recentTasks = Task::with(['assignedUser', 'createdByUser'])
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get();

        $teamMembers = User::where('role', '!=', 'super_admin')
            ->withCount([
                'assignedTasks',
                'assignedTasks as completed_tasks_count' => function ($q) {
                    $q->whereIn('status', ['completed', 'approved']);
                },
                'assignedTasks as in_progress_count' => function ($q) {
                    $q->where('status', 'in_progress');
                },
            ])
            ->get();

        $allTasks = Task::with('assignedUser')
            ->orderBy('planned_start')
            ->get();

        return view('dashboard.admin', compact('stats', 'recentTasks', 'teamMembers', 'allTasks'));
    }

    private function marketingDashboard()
    {
        $user = Auth::user();

        $stats = [
            'my_tasks'       => Task::where('assigned_to', $user->id)->count(),
            'pending_review' => Task::where('status', 'under_review')->count(),
            'approved_today' => TaskApproval::where('reviewed_by', $user->id)
                ->whereDate('created_at', today())
                ->where('status', 'approved')
                ->count(),
            'my_pending'     => Task::where('assigned_to', $user->id)
                ->where('status', 'pending')
                ->count(),
        ];

        $pendingApprovals = Task::with(['assignedUser', 'createdByUser'])
            ->where('status', 'under_review')
            ->orderBy('updated_at', 'desc')
            ->get();

        $myTasks = Task::where('assigned_to', $user->id)
            ->orderBy('planned_start')
            ->get();

        return view('dashboard.marketing', compact('stats', 'pendingApprovals', 'myTasks'));
    }

    private function employeeDashboard()
    {
        $user = Auth::user();

        $stats = [
            'total'       => Task::where('assigned_to', $user->id)->count(),
            'pending'     => Task::where('assigned_to', $user->id)->where('status', 'pending')->count(),
            'in_progress' => Task::where('assigned_to', $user->id)->where('status', 'in_progress')->count(),
            'completed'   => Task::where('assigned_to', $user->id)->whereIn('status', ['completed', 'approved'])->count(),
        ];

        $tasks = Task::where('assigned_to', $user->id)
            ->with('createdByUser')
            ->orderByRaw("FIELD(status, 'in_progress', 'pending', 'under_review', 'rejected', 'completed', 'approved')")
            ->orderBy('planned_end')
            ->get();

        return view('dashboard.employee', compact('stats', 'tasks'));
    }
}
