<?php

namespace App\Helpers;

use App\Models\Notification;
use App\Models\User;

class NotificationHelper
{
    /**
     * Send a notification to a specific user.
     */
    public static function send(int $userId, string $type, string $title, string $message, array $data = []): void
    {
        Notification::create([
            'user_id' => $userId,
            'type'    => $type,
            'title'   => $title,
            'message' => $message,
            'data'    => $data ?: null,
        ]);
    }

    /**
     * Notify all admin + super_admin users (skip a specific user if needed).
     */
    public static function notifyAdmins(string $type, string $title, string $message, array $data = [], int $excludeUserId = 0): void
    {
        $admins = User::whereIn('role', ['super_admin', 'admin'])
            ->where('status', 'active')
            ->when($excludeUserId, fn($q) => $q->where('id', '!=', $excludeUserId))
            ->get();

        foreach ($admins as $admin) {
            static::send($admin->id, $type, $title, $message, $data);
        }
    }

    /**
     * Notify admins + super_admin + marketing (skip a specific user if needed).
     */
    public static function notifyManagement(string $type, string $title, string $message, array $data = [], int $excludeUserId = 0): void
    {
        $users = User::whereIn('role', ['super_admin', 'admin', 'marketing'])
            ->where('status', 'active')
            ->when($excludeUserId, fn($q) => $q->where('id', '!=', $excludeUserId))
            ->get();

        foreach ($users as $user) {
            static::send($user->id, $type, $title, $message, $data);
        }
    }
}
